<?php
defined( 'ABSPATH' ) || exit;

class HRM_Reports {

    public static function init() {
        add_action( 'wp_ajax_hrm_generate_report', [ __CLASS__, 'handle_ajax_generate_report' ] );
        add_action( 'wp_ajax_hrm_export_report', [ __CLASS__, 'handle_ajax_export_report' ] );
        add_action( 'wp_ajax_hrm_get_report_data', [ __CLASS__, 'handle_ajax_get_report_data' ] );
    }

    /**
     * Render the main reports dashboard
     */
    public static function render_reports() {
        if ( ! current_user_can( 'manage_options' ) ) {
            wp_die( 'Unauthorized access' );
        }

        ?>
        <div class="hrm-reports-wrapper">
            <!-- Header Section -->
            <div class="hrm-reports-header">
                <div class="header-left">
                    <h1 class="page-title">
                        <span class="dashicons dashicons-chart-bar"></span>
                        Reports & Analytics
                    </h1>
                    <p class="page-subtitle">Comprehensive insights into your HR operations</p>
                </div>
                <div class="header-right">
                    <div class="quick-employee-filter">
                        <label for="quick-employee-filter">
                            <span class="dashicons dashicons-admin-users"></span>
                            Quick Filter:
                        </label>
                        <select id="quick-employee-filter" class="employee-select-dropdown">
                            <option value="">All Employees</option>
                            <?php self::render_employee_options(); ?>
                        </select>
                    </div>
                    <button class="hrm-btn hrm-btn-secondary" id="refresh-reports">
                        <span class="dashicons dashicons-update"></span>
                        Refresh
                    </button>
                </div>
            </div>

            <!-- Quick Stats Cards -->
            <div class="hrm-stats-grid">
                <?php self::render_quick_stats(); ?>
            </div>

            <!-- Report Type Selection -->
            <div class="hrm-report-selector">
                <div class="report-selector-header">
                    <h2><span class="dashicons dashicons-analytics"></span> Generate Report</h2>
                </div>
                <div class="report-selector-content">
                    <div class="report-type-grid">
                        <div class="report-type-card" data-report="attendance">
                            <div class="report-icon attendance-icon">
                                <span class="dashicons dashicons-calendar-alt"></span>
                            </div>
                            <h3>Attendance Report</h3>
                            <p>Complete attendance analysis with trends and patterns</p>
                            <button class="hrm-btn hrm-btn-primary select-report-btn">Generate</button>
                        </div>

                        <div class="report-type-card" data-report="performance">
                            <div class="report-icon performance-icon">
                                <span class="dashicons dashicons-chart-line"></span>
                            </div>
                            <h3>Performance Report</h3>
                            <p>Task completion rates and productivity metrics</p>
                            <button class="hrm-btn hrm-btn-primary select-report-btn">Generate</button>
                        </div>

                        <div class="report-type-card" data-report="employee">
                            <div class="report-icon employee-icon">
                                <span class="dashicons dashicons-groups"></span>
                            </div>
                            <h3>Employee Report</h3>
                            <p>Comprehensive employee data and statistics</p>
                            <button class="hrm-btn hrm-btn-primary select-report-btn">Generate</button>
                        </div>

                        <div class="report-type-card" data-report="tasks">
                            <div class="report-icon tasks-icon">
                                <span class="dashicons dashicons-list-view"></span>
                            </div>
                            <h3>Tasks Report</h3>
                            <p>Task distribution, status, and completion analysis</p>
                            <button class="hrm-btn hrm-btn-primary select-report-btn">Generate</button>
                        </div>

                        <div class="report-type-card" data-report="daily-progress">
                            <div class="report-icon progress-icon">
                                <span class="dashicons dashicons-admin-page"></span>
                            </div>
                            <h3>Daily Progress Report</h3>
                            <p>Daily task updates and work logs analysis</p>
                            <button class="hrm-btn hrm-btn-primary select-report-btn">Generate</button>
                        </div>

                        <div class="report-type-card" data-report="leave">
                            <div class="report-icon leave-icon">
                                <span class="dashicons dashicons-calendar"></span>
                            </div>
                            <h3>Leave Report</h3>
                            <p>Complete leave analysis with employee leave history and status</p>
                            <button class="hrm-btn hrm-btn-primary select-report-btn">Generate</button>
                        </div>

                        <div class="report-type-card" data-report="summary">
                            <div class="report-icon summary-icon">
                                <span class="dashicons dashicons-portfolio"></span>
                            </div>
                            <h3>Executive Summary</h3>
                            <p>High-level overview of all HR metrics</p>
                            <button class="hrm-btn hrm-btn-primary select-report-btn">Generate</button>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Report Configuration Panel (Hidden by default) -->
            <div class="hrm-report-config" id="report-config-panel" style="display: none;">
                <div class="config-header">
                    <h2 id="config-title">Configure Report</h2>
                    <button class="close-config-btn">
                        <span class="dashicons dashicons-no-alt"></span>
                    </button>
                </div>
                <div class="config-content">
                    <form id="report-config-form">
                        <input type="hidden" id="report-type" name="report_type" value="">
                        
                        <div class="config-row">
                            <div class="config-field">
                                <label for="date-from">From Date</label>
                                <input type="date" id="date-from" name="date_from" required>
                            </div>
                            <div class="config-field">
                                <label for="date-to">To Date</label>
                                <input type="date" id="date-to" name="date_to" required>
                            </div>
                        </div>

                        <div class="config-row" id="employee-select-row">
                            <div class="config-field full-width">
                                <label for="employee-filter">
                                    <span class="dashicons dashicons-admin-users"></span>
                                    Select Employee (Optional)
                                </label>
                                <select id="employee-filter" name="employee_id" class="employee-select-dropdown">
                                    <option value="">All Employees</option>
                                    <?php self::render_employee_options(); ?>
                                </select>
                            </div>
                        </div>

                        <div class="config-row" id="status-select-row" style="display: none;">
                            <div class="config-field full-width">
                                <label for="status-filter">Filter by Status</label>
                                <select id="status-filter" name="status">
                                    <option value="">All Statuses</option>
                                    <option value="pending">Pending</option>
                                    <option value="in_progress">In Progress</option>
                                    <option value="completed">Completed</option>
                                </select>
                            </div>
                        </div>

                        <div class="config-row" id="leave-type-select-row" style="display: none;">
                            <div class="config-field">
                                <label for="leave-type-filter">
                                    <span class="dashicons dashicons-category"></span>
                                    Leave Type
                                </label>
                                <select id="leave-type-filter" name="leave_type" class="employee-select-dropdown">
                                    <option value="">All Types</option>
                                    <option value="casual">Casual Leave</option>
                                    <option value="sick">Sick Leave</option>
                                    <option value="annual">Annual Leave</option>
                                    <option value="unpaid">Unpaid Leave</option>
                                </select>
                            </div>
                            <div class="config-field">
                                <label for="leave-status-filter">
                                    <span class="dashicons dashicons-tag"></span>
                                    Leave Status
                                </label>
                                <select id="leave-status-filter" name="leave_status" class="employee-select-dropdown">
                                    <option value="">All Statuses</option>
                                    <option value="pending">Pending</option>
                                    <option value="approved">Approved</option>
                                    <option value="rejected">Rejected</option>
                                </select>
                            </div>
                        </div>

                        <div class="config-actions">
                            <button type="button" class="hrm-btn hrm-btn-secondary close-config-btn">Cancel</button>
                            <button type="submit" class="hrm-btn hrm-btn-primary" id="generate-report-btn">
                                <span class="dashicons dashicons-chart-bar"></span>
                                Generate Report
                            </button>
                        </div>
                    </form>
                </div>
            </div>

            <!-- Report Results Section -->
            <div class="hrm-report-results" id="report-results" style="display: none;">
                <div class="results-header">
                    <div class="results-header-left">
                        <h2 id="results-title">Report Results</h2>
                        <p id="results-subtitle"></p>
                    </div>
                    <div class="results-header-right">
                        <button class="hrm-btn hrm-btn-secondary" id="print-report">
                            <span class="dashicons dashicons-printer"></span>
                            Print
                        </button>
                        <button class="hrm-btn hrm-btn-secondary" id="export-pdf">
                            <span class="dashicons dashicons-pdf"></span>
                            Export PDF
                        </button>
                        <button class="hrm-btn hrm-btn-primary" id="export-excel">
                            <span class="dashicons dashicons-download"></span>
                            Export Excel
                        </button>
                    </div>
                </div>
                <div class="results-content" id="results-content">
                    <!-- Report content will be dynamically inserted here -->
                </div>
            </div>
        </div>
        <?php
    }

    /**
     * Render quick statistics cards
     */
    private static function render_quick_stats() {
        global $wpdb;
        
        // Total Employees
        $total_employees = $wpdb->get_var( "
            SELECT COUNT(DISTINCT ID) 
            FROM {$wpdb->users} u
            INNER JOIN {$wpdb->usermeta} um ON u.ID = um.user_id
            WHERE um.meta_key = 'wp_capabilities'
            AND um.meta_value LIKE '%hrm_employee%'
        " );

        // Today's Attendance
        $today = current_time( 'Y-m-d' );
        $today_attendance = $wpdb->get_var( $wpdb->prepare( "
            SELECT COUNT(*) 
            FROM {$wpdb->prefix}hrm_attendance 
            WHERE date = %s AND status = 'present'
        ", $today ) );

        // Active Tasks
        $active_tasks = $wpdb->get_var( "
            SELECT COUNT(*) 
            FROM {$wpdb->prefix}hrm_tasks 
            WHERE status IN ('pending', 'in_progress')
        " );

        // This Month's Progress Entries
        $month_start = date( 'Y-m-01' );
        $month_end = date( 'Y-m-t' );
        $progress_entries = $wpdb->get_var( $wpdb->prepare( "
            SELECT COUNT(*) 
            FROM {$wpdb->prefix}hrm_daily_progress 
            WHERE date BETWEEN %s AND %s
        ", $month_start, $month_end ) );

        ?>
        <div class="stats-card stats-employees">
            <div class="stats-icon">
                <span class="dashicons dashicons-groups"></span>
            </div>
            <div class="stats-content">
                <h3>Total Employees</h3>
                <p class="stats-number"><?php echo number_format( $total_employees ?: 0 ); ?></p>
                <span class="stats-label">Active workforce</span>
            </div>
        </div>

        <div class="stats-card stats-attendance">
            <div class="stats-icon">
                <span class="dashicons dashicons-yes-alt"></span>
            </div>
            <div class="stats-content">
                <h3>Today's Present</h3>
                <p class="stats-number"><?php echo number_format( $today_attendance ?: 0 ); ?></p>
                <span class="stats-label">Out of <?php echo $total_employees ?: 0; ?> employees</span>
            </div>
        </div>

        <div class="stats-card stats-tasks">
            <div class="stats-icon">
                <span class="dashicons dashicons-list-view"></span>
            </div>
            <div class="stats-content">
                <h3>Active Tasks</h3>
                <p class="stats-number"><?php echo number_format( $active_tasks ?: 0 ); ?></p>
                <span class="stats-label">Pending & In Progress</span>
            </div>
        </div>

        <div class="stats-card stats-progress">
            <div class="stats-icon">
                <span class="dashicons dashicons-chart-area"></span>
            </div>
            <div class="stats-content">
                <h3>This Month's Updates</h3>
                <p class="stats-number"><?php echo number_format( $progress_entries ?: 0 ); ?></p>
                <span class="stats-label">Progress entries logged</span>
            </div>
        </div>
        <?php
    }

    /**
     * Render employee options for dropdown
     */
    private static function render_employee_options() {
        global $wpdb;
        
        // Query employees from custom post type (more reliable than querying users)
        $employees_posts = get_posts([
            'post_type' => 'hrm_employee', 
            'posts_per_page' => -1,
            'post_status' => ['publish', 'draft'],
            'orderby' => 'title',
            'order' => 'ASC'
        ]);
        
        // If we have employees from post type, use those
        if ( !empty($employees_posts) ) {
            $added_users = []; // Track which user IDs we've already added
            
            foreach ( $employees_posts as $emp_post ) {
                // Get the WordPress user ID associated with this employee
                $user_id = $emp_post->post_author;
                
                // Skip if we've already added this user (prevents duplicates)
                if ( in_array($user_id, $added_users) ) {
                    continue;
                }
                
                // Display employee name with user ID as value
                echo '<option value="' . esc_attr( $user_id ) . '">' . esc_html( $emp_post->post_title ) . '</option>';
                
                // Mark this user as added
                $added_users[] = $user_id;
            }
        } else {
            // Fallback: Query users with hrm_employee role
            $users = get_users([
                'role' => 'hrm_employee',
                'orderby' => 'display_name',
                'order' => 'ASC'
            ]);
            
            foreach ( $users as $user ) {
                echo '<option value="' . esc_attr( $user->ID ) . '">' . esc_html( $user->display_name ) . '</option>';
            }
        }
    }

    /**
     * Handle AJAX request to generate report
     */
    public static function handle_ajax_generate_report() {
        check_ajax_referer( 'hrm_ajax_nonce', 'nonce' );
        
        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( 'Unauthorized access' );
        }

        $report_type = sanitize_text_field( $_POST['report_type'] ?? '' );
        $date_from = sanitize_text_field( $_POST['date_from'] ?? '' );
        $date_to = sanitize_text_field( $_POST['date_to'] ?? '' );
        $employee_id = intval( $_POST['employee_id'] ?? 0 );
        $status = sanitize_text_field( $_POST['status'] ?? '' );
        $leave_type = sanitize_text_field( $_POST['leave_type'] ?? '' );
        $leave_status = sanitize_text_field( $_POST['leave_status'] ?? '' );

        $report_data = self::generate_report_data( $report_type, $date_from, $date_to, $employee_id, $status, $leave_type, $leave_status );

        if ( $report_data ) {
            wp_send_json_success( $report_data );
        } else {
            wp_send_json_error( 'Failed to generate report' );
        }
    }

    /**
     * Generate report data based on type
     */
    private static function generate_report_data( $type, $date_from, $date_to, $employee_id, $status, $leave_type = '', $leave_status = '' ) {
        switch ( $type ) {
            case 'attendance':
                return self::generate_attendance_report( $date_from, $date_to, $employee_id );
            case 'performance':
                return self::generate_performance_report( $date_from, $date_to, $employee_id );
            case 'employee':
                return self::generate_employee_report( $date_from, $date_to, $employee_id );
            case 'tasks':
                return self::generate_tasks_report( $date_from, $date_to, $employee_id, $status );
            case 'daily-progress':
                return self::generate_daily_progress_report( $date_from, $date_to, $employee_id );
            case 'leave':
                return self::generate_leave_report( $date_from, $date_to, $employee_id, $leave_type, $leave_status );
            case 'summary':
                return self::generate_summary_report( $date_from, $date_to );
            default:
                return false;
        }
    }

    /**
     * Generate Attendance Report
     */
    private static function generate_attendance_report( $date_from, $date_to, $employee_id ) {
        global $wpdb;

        $where = $wpdb->prepare( "WHERE date BETWEEN %s AND %s", $date_from, $date_to );
        if ( $employee_id ) {
            $where .= $wpdb->prepare( " AND employee_id = %d", $employee_id );
        }

        // Get attendance records
        $records = $wpdb->get_results( "
            SELECT a.*, u.display_name as employee_name
            FROM {$wpdb->prefix}hrm_attendance a
            LEFT JOIN {$wpdb->users} u ON a.employee_id = u.ID
            {$where}
            ORDER BY a.date DESC, u.display_name ASC
        " );

        // Calculate statistics
        $total_records = count( $records );
        $present_count = count( array_filter( $records, fn($r) => in_array($r->status, ['present', 'late']) ) );
        $half_day_count = count( array_filter( $records, fn($r) => $r->status === 'half_day' ) );
        $leave_count = count( array_filter( $records, fn($r) => $r->status === 'leave' ) );
        $absent_count = count( array_filter( $records, fn($r) => $r->status === 'absent' ) );
        $late_count = count( array_filter( $records, fn($r) => $r->status === 'late' ) );
        $total_hours = array_sum( array_column( $records, 'hours_worked' ) );

        // Build subtitle with employee name if filtered
        $subtitle = sprintf( 'From %s to %s', date( 'M d, Y', strtotime( $date_from ) ), date( 'M d, Y', strtotime( $date_to ) ) );
        if ( $employee_id && !empty( $records ) ) {
            $subtitle .= ' - ' . $records[0]->employee_name;
        }

        return [
            'type' => 'attendance',
            'title' => 'Attendance Report',
            'subtitle' => $subtitle,
            'stats' => [
                'total_records' => $total_records,
                'present' => $present_count,
                'absent' => $absent_count,
                'late' => $late_count,
                'half_day' => $half_day_count,
                'leave' => $leave_count,
                'total_hours' => round( $total_hours, 2 ),
                'average_hours' => $present_count > 0 ? round( $total_hours / $present_count, 2 ) : 0,
                'attendance_rate' => $total_records > 0 ? round( ( $present_count / $total_records ) * 100, 2 ) : 0
            ],
            'records' => $records,
            'chart_data' => self::prepare_attendance_chart_data( $records )
        ];
    }

    /**
     * Prepare attendance chart data
     */
    private static function prepare_attendance_chart_data( $records ) {
        $status_counts = [
            'present' => 0,
            'absent' => 0,
            'late' => 0,
            'half_day' => 0,
            'leave' => 0
        ];

        foreach ( $records as $record ) {
            if ( isset( $status_counts[$record->status] ) ) {
                $status_counts[$record->status]++;
            }
        }

        return $status_counts;
    }

    /**
     * Generate Performance Report
     */
    private static function generate_performance_report( $date_from, $date_to, $employee_id ) {
        global $wpdb;

        $where = $wpdb->prepare( "WHERE t.created_at BETWEEN %s AND %s", $date_from . ' 00:00:00', $date_to . ' 23:59:59' );
        if ( $employee_id ) {
            $where .= $wpdb->prepare( " AND t.assigned_to = %d", $employee_id );
        }

        // Get task statistics
        $tasks = $wpdb->get_results( "
            SELECT t.*, u.display_name as employee_name
            FROM {$wpdb->prefix}hrm_tasks t
            LEFT JOIN {$wpdb->users} u ON t.assigned_to = u.ID
            {$where}
        " );

        // Get daily progress statistics
        $progress_where = str_replace( 't.created_at', 'dp.date', $where );
        $progress_where = str_replace( 't.assigned_to', 'dp.employee_id', $progress_where );
        
        $progress_stats = $wpdb->get_results( "
            SELECT 
                dp.employee_id,
                u.display_name as employee_name,
                COUNT(*) as total_entries,
                SUM(dp.hours_logged) as total_hours,
                AVG(dp.hours_logged) as avg_hours
            FROM {$wpdb->prefix}hrm_daily_progress dp
            LEFT JOIN {$wpdb->users} u ON dp.employee_id = u.ID
            {$progress_where}
            GROUP BY dp.employee_id
            ORDER BY total_hours DESC
        " );

        $total_tasks = count( $tasks );
        $completed = count( array_filter( $tasks, fn($t) => $t->status === 'completed' ) );
        $in_progress = count( array_filter( $tasks, fn($t) => $t->status === 'in_progress' ) );
        $pending = count( array_filter( $tasks, fn($t) => $t->status === 'pending' ) );

        // Build subtitle with employee name if filtered
        $subtitle = sprintf( 'From %s to %s', date( 'M d, Y', strtotime( $date_from ) ), date( 'M d, Y', strtotime( $date_to ) ) );
        if ( $employee_id && !empty( $tasks ) ) {
            $subtitle .= ' - ' . $tasks[0]->employee_name;
        }

        return [
            'type' => 'performance',
            'title' => 'Performance Report',
            'subtitle' => $subtitle,
            'stats' => [
                'total_tasks' => $total_tasks,
                'completed' => $completed,
                'in_progress' => $in_progress,
                'pending' => $pending,
                'completion_rate' => $total_tasks > 0 ? round( ( $completed / $total_tasks ) * 100, 2 ) : 0
            ],
            'tasks' => $tasks,
            'progress_stats' => $progress_stats,
            'chart_data' => [
                'completed' => $completed,
                'in_progress' => $in_progress,
                'pending' => $pending
            ]
        ];
    }

    /**
     * Generate Employee Report
     */
    private static function generate_employee_report( $date_from, $date_to, $employee_id ) {
        global $wpdb;

        $where_clause = '';
        if ( $employee_id ) {
            $where_clause = $wpdb->prepare( "AND u.ID = %d", $employee_id );
        }

        $employees = $wpdb->get_results( "
            SELECT 
                u.ID,
                u.display_name,
                u.user_email,
                u.user_registered
            FROM {$wpdb->users} u
            INNER JOIN {$wpdb->usermeta} um ON u.ID = um.user_id
            WHERE um.meta_key = 'wp_capabilities'
            AND um.meta_value LIKE '%hrm_employee%'
            {$where_clause}
            ORDER BY u.display_name ASC
        " );

        $employee_stats = [];
        foreach ( $employees as $employee ) {
            // Get attendance stats
            $attendance_stats = $wpdb->get_row( $wpdb->prepare( "
                SELECT 
                    COUNT(*) as total_days,
                    SUM(CASE WHEN status = 'present' THEN 1 ELSE 0 END) as present_days,
                    SUM(hours_worked) as total_hours
                FROM {$wpdb->prefix}hrm_attendance
                WHERE employee_id = %d AND date BETWEEN %s AND %s
            ", $employee->ID, $date_from, $date_to ) );

            // Get task stats
            $task_stats = $wpdb->get_row( $wpdb->prepare( "
                SELECT 
                    COUNT(*) as total_tasks,
                    SUM(CASE WHEN status = 'completed' THEN 1 ELSE 0 END) as completed_tasks
                FROM {$wpdb->prefix}hrm_tasks
                WHERE assigned_to = %d 
                AND created_at BETWEEN %s AND %s
            ", $employee->ID, $date_from . ' 00:00:00', $date_to . ' 23:59:59' ) );

            // Get progress entries
            $progress_count = $wpdb->get_var( $wpdb->prepare( "
                SELECT COUNT(*)
                FROM {$wpdb->prefix}hrm_daily_progress
                WHERE employee_id = %d AND date BETWEEN %s AND %s
            ", $employee->ID, $date_from, $date_to ) );

            $employee_stats[] = [
                'employee' => $employee,
                'attendance' => $attendance_stats,
                'tasks' => $task_stats,
                'progress_entries' => $progress_count
            ];
        }

        // Build subtitle with employee name if filtered
        $subtitle = sprintf( 'From %s to %s', date( 'M d, Y', strtotime( $date_from ) ), date( 'M d, Y', strtotime( $date_to ) ) );
        if ( $employee_id && !empty( $employees ) ) {
            $subtitle .= ' - ' . $employees[0]->display_name;
        }

        return [
            'type' => 'employee',
            'title' => 'Employee Report',
            'subtitle' => $subtitle,
            'employees' => $employee_stats,
            'total_employees' => count( $employees )
        ];
    }

    /**
     * Generate Tasks Report
     */
    private static function generate_tasks_report( $date_from, $date_to, $employee_id, $status ) {
        global $wpdb;

        $where = $wpdb->prepare( "WHERE created_at BETWEEN %s AND %s", $date_from . ' 00:00:00', $date_to . ' 23:59:59' );
        if ( $employee_id ) {
            $where .= $wpdb->prepare( " AND assigned_to = %d", $employee_id );
        }
        if ( $status ) {
            $where .= $wpdb->prepare( " AND status = %s", $status );
        }

        $tasks = $wpdb->get_results( "
            SELECT t.*, u.display_name as employee_name
            FROM {$wpdb->prefix}hrm_tasks t
            LEFT JOIN {$wpdb->users} u ON t.assigned_to = u.ID
            {$where}
            ORDER BY t.created_at DESC
        " );

        $total_tasks = count( $tasks );
        $completed = count( array_filter( $tasks, fn($t) => $t->status === 'completed' ) );
        $in_progress = count( array_filter( $tasks, fn($t) => $t->status === 'in_progress' ) );
        $pending = count( array_filter( $tasks, fn($t) => $t->status === 'pending' ) );

        // Task distribution by employee
        $employee_distribution = [];
        foreach ( $tasks as $task ) {
            if ( ! isset( $employee_distribution[$task->assigned_to] ) ) {
                $employee_distribution[$task->assigned_to] = [
                    'name' => $task->employee_name,
                    'total' => 0,
                    'completed' => 0,
                    'in_progress' => 0,
                    'pending' => 0
                ];
            }
            $employee_distribution[$task->assigned_to]['total']++;
            $employee_distribution[$task->assigned_to][$task->status]++;
        }

        // Build subtitle with employee name if filtered
        $subtitle = sprintf( 'From %s to %s', date( 'M d, Y', strtotime( $date_from ) ), date( 'M d, Y', strtotime( $date_to ) ) );
        if ( $employee_id && !empty( $tasks ) ) {
            $subtitle .= ' - ' . $tasks[0]->employee_name;
        }

        return [
            'type' => 'tasks',
            'title' => 'Tasks Report',
            'subtitle' => $subtitle,
            'stats' => [
                'total_tasks' => $total_tasks,
                'completed' => $completed,
                'in_progress' => $in_progress,
                'pending' => $pending,
                'completion_rate' => $total_tasks > 0 ? round( ( $completed / $total_tasks ) * 100, 2 ) : 0
            ],
            'tasks' => $tasks,
            'employee_distribution' => array_values( $employee_distribution ),
            'chart_data' => [
                'completed' => $completed,
                'in_progress' => $in_progress,
                'pending' => $pending
            ]
        ];
    }

    /**
     * Generate Daily Progress Report
     */
    private static function generate_daily_progress_report( $date_from, $date_to, $employee_id ) {
        global $wpdb;

        $where = $wpdb->prepare( "WHERE dp.date BETWEEN %s AND %s", $date_from, $date_to );
        if ( $employee_id ) {
            $where .= $wpdb->prepare( " AND dp.employee_id = %d", $employee_id );
        }

        $progress_entries = $wpdb->get_results( "
            SELECT dp.*, u.display_name as employee_name, t.title as task_title
            FROM {$wpdb->prefix}hrm_daily_progress dp
            LEFT JOIN {$wpdb->users} u ON dp.employee_id = u.ID
            LEFT JOIN {$wpdb->prefix}hrm_tasks t ON dp.task_id = t.id
            {$where}
            ORDER BY dp.date DESC, u.display_name ASC
        " );

        $total_entries = count( $progress_entries );
        $total_hours = array_sum( array_column( $progress_entries, 'hours_logged' ) );
        $in_progress = count( array_filter( $progress_entries, fn($p) => $p->status === 'in_progress' ) );
        $completed = count( array_filter( $progress_entries, fn($p) => $p->status === 'completed' ) );

        // Build subtitle with employee name if filtered
        $subtitle = sprintf( 'From %s to %s', date( 'M d, Y', strtotime( $date_from ) ), date( 'M d, Y', strtotime( $date_to ) ) );
        if ( $employee_id && !empty( $progress_entries ) ) {
            $subtitle .= ' - ' . $progress_entries[0]->employee_name;
        }

        return [
            'type' => 'daily-progress',
            'title' => 'Daily Progress Report',
            'subtitle' => $subtitle,
            'stats' => [
                'total_entries' => $total_entries,
                'total_hours' => round( $total_hours, 2 ),
                'in_progress' => $in_progress,
                'completed' => $completed,
                'average_hours' => $total_entries > 0 ? round( $total_hours / $total_entries, 2 ) : 0
            ],
            'entries' => $progress_entries,
            'chart_data' => [
                'in_progress' => $in_progress,
                'completed' => $completed
            ]
        ];
    }

    /**
     * Generate Leave Report
     */
    private static function generate_leave_report( $date_from, $date_to, $employee_id, $leave_type = '', $leave_status = '' ) {
        global $wpdb;
        $table_name = $wpdb->prefix . 'hrm_leave';

        // Build WHERE clause
        $where = $wpdb->prepare( "WHERE (start_date BETWEEN %s AND %s OR end_date BETWEEN %s AND %s OR (start_date <= %s AND end_date >= %s))", 
            $date_from, $date_to, $date_from, $date_to, $date_from, $date_to );
        
        if ( $employee_id ) {
            $where .= $wpdb->prepare( " AND employee_id = %d", $employee_id );
        }
        
        if ( $leave_type ) {
            $where .= $wpdb->prepare( " AND leave_type = %s", $leave_type );
        }
        
        if ( $leave_status ) {
            $where .= $wpdb->prepare( " AND status = %s", $leave_status );
        }

        // Get leave records
        $leave_records = $wpdb->get_results( "
            SELECT l.*, u.display_name as employee_name
            FROM {$table_name} l
            LEFT JOIN {$wpdb->users} u ON l.employee_id = u.ID
            {$where}
            ORDER BY l.start_date DESC, u.display_name ASC
        " );

        // Calculate total days for each leave record
        foreach ( $leave_records as &$record ) {
            $start = new \DateTime( $record->start_date );
            $end = new \DateTime( $record->end_date );
            $record->total_days = $end->diff( $start )->days + 1;
        }

        // Calculate statistics
        $total_records = count( $leave_records );
        $pending_count = count( array_filter( $leave_records, fn($r) => $r->status === 'pending' ) );
        $approved_count = count( array_filter( $leave_records, fn($r) => $r->status === 'approved' ) );
        $rejected_count = count( array_filter( $leave_records, fn($r) => $r->status === 'rejected' ) );
        $total_leave_days = array_sum( array_column( $leave_records, 'total_days' ) );

        // Calculate leave by type
        $leave_by_type = [
            'casual' => count( array_filter( $leave_records, fn($r) => $r->leave_type === 'casual' ) ),
            'sick' => count( array_filter( $leave_records, fn($r) => $r->leave_type === 'sick' ) ),
            'annual' => count( array_filter( $leave_records, fn($r) => $r->leave_type === 'annual' ) ),
            'unpaid' => count( array_filter( $leave_records, fn($r) => $r->leave_type === 'unpaid' ) )
        ];

        // Get unique employees count
        $unique_employees = count( array_unique( array_column( $leave_records, 'employee_id' ) ) );

        // Build subtitle with employee name if filtered
        $subtitle = sprintf( 'From %s to %s', date( 'M d, Y', strtotime( $date_from ) ), date( 'M d, Y', strtotime( $date_to ) ) );
        if ( $employee_id && !empty( $leave_records ) ) {
            $subtitle .= ' - ' . $leave_records[0]->employee_name;
        }
        if ( $leave_type ) {
            $subtitle .= ' - ' . ucfirst( $leave_type ) . ' Leave';
        }
        if ( $leave_status ) {
            $subtitle .= ' (' . ucfirst( $leave_status ) . ')';
        }

        return [
            'type' => 'leave',
            'title' => 'Leave Report',
            'subtitle' => $subtitle,
            'stats' => [
                'total_requests' => $total_records,
                'pending' => $pending_count,
                'approved' => $approved_count,
                'rejected' => $rejected_count,
                'total_leave_days' => $total_leave_days,
                'unique_employees' => $unique_employees,
                'approval_rate' => $total_records > 0 ? round( ( $approved_count / $total_records ) * 100, 1 ) : 0
            ],
            'leave_by_type' => $leave_by_type,
            'records' => $leave_records,
            'chart_data' => [
                'pending' => $pending_count,
                'approved' => $approved_count,
                'rejected' => $rejected_count
            ]
        ];
    }

    /**
     * Generate Executive Summary Report
     */
    private static function generate_summary_report( $date_from, $date_to ) {
        global $wpdb;

        // Get all basic stats
        $attendance_report = self::generate_attendance_report( $date_from, $date_to, 0 );
        $performance_report = self::generate_performance_report( $date_from, $date_to, 0 );
        $tasks_report = self::generate_tasks_report( $date_from, $date_to, 0, '' );
        $progress_report = self::generate_daily_progress_report( $date_from, $date_to, 0 );

        // Top performers by task completion
        $top_performers = $wpdb->get_results( $wpdb->prepare( "
            SELECT 
                u.display_name as employee_name,
                COUNT(*) as completed_tasks,
                SUM(dp.hours_logged) as total_hours
            FROM {$wpdb->prefix}hrm_tasks t
            LEFT JOIN {$wpdb->users} u ON t.assigned_to = u.ID
            LEFT JOIN {$wpdb->prefix}hrm_daily_progress dp ON t.assigned_to = dp.employee_id
            WHERE t.status = 'completed'
            AND t.created_at BETWEEN %s AND %s
            AND (dp.date BETWEEN %s AND %s OR dp.date IS NULL)
            GROUP BY t.assigned_to
            ORDER BY completed_tasks DESC, total_hours DESC
            LIMIT 5
        ", $date_from . ' 00:00:00', $date_to . ' 23:59:59', $date_from, $date_to ) );

        return [
            'type' => 'summary',
            'title' => 'Executive Summary Report',
            'subtitle' => sprintf( 'From %s to %s', date( 'M d, Y', strtotime( $date_from ) ), date( 'M d, Y', strtotime( $date_to ) ) ),
            'attendance' => $attendance_report['stats'],
            'performance' => $performance_report['stats'],
            'tasks' => $tasks_report['stats'],
            'progress' => $progress_report['stats'],
            'top_performers' => $top_performers
        ];
    }

    /**
     * Handle AJAX request to export report
     */
    public static function handle_ajax_export_report() {
        check_ajax_referer( 'hrm_ajax_nonce', 'nonce' );
        
        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( 'Unauthorized access' );
        }

        $report_type = sanitize_text_field( $_POST['report_type'] ?? '' );
        $format = sanitize_text_field( $_POST['format'] ?? 'excel' );
        $date_from = sanitize_text_field( $_POST['date_from'] ?? '' );
        $date_to = sanitize_text_field( $_POST['date_to'] ?? '' );
        $employee_id = intval( $_POST['employee_id'] ?? 0 );
        $status = sanitize_text_field( $_POST['status'] ?? '' );
        $leave_type = sanitize_text_field( $_POST['leave_type'] ?? '' );
        $leave_status = sanitize_text_field( $_POST['leave_status'] ?? '' );

        $report_data = self::generate_report_data( $report_type, $date_from, $date_to, $employee_id, $status, $leave_type, $leave_status );

        if ( $format === 'excel' ) {
            self::export_to_excel( $report_type, $report_data );
        } else {
            self::export_to_pdf( $report_type, $report_data );
        }
    }

    /**
     * Export report to Excel (CSV format)
     */
    private static function export_to_excel( $type, $data ) {
        $filename = sprintf( 'hrm_%s_report_%s.csv', $type, date( 'Y-m-d' ) );
        
        header( 'Content-Type: text/csv; charset=utf-8' );
        header( 'Content-Disposition: attachment; filename=' . $filename );
        
        $output = fopen( 'php://output', 'w' );
        
        // Write BOM for UTF-8
        fprintf( $output, chr(0xEF).chr(0xBB).chr(0xBF) );

        switch ( $type ) {
            case 'attendance':
                fputcsv( $output, [ 'Employee', 'Date', 'Check In', 'Check Out', 'Hours Worked', 'Status' ] );
                foreach ( $data['records'] as $record ) {
                    fputcsv( $output, [
                        $record->employee_name,
                        $record->date,
                        $record->check_in,
                        $record->check_out,
                        $record->hours_worked,
                        ucfirst( $record->status )
                    ] );
                }
                break;

            case 'tasks':
                fputcsv( $output, [ 'Task Title', 'Assigned To', 'Due Date', 'Status', 'Created At' ] );
                foreach ( $data['tasks'] as $task ) {
                    fputcsv( $output, [
                        $task->title,
                        $task->employee_name,
                        $task->due_date,
                        ucfirst( str_replace( '_', ' ', $task->status ) ),
                        $task->created_at
                    ] );
                }
                break;

            case 'daily-progress':
                fputcsv( $output, [ 'Employee', 'Date', 'Task', 'Hours Logged', 'Description', 'Status' ] );
                foreach ( $data['entries'] as $entry ) {
                    fputcsv( $output, [
                        $entry->employee_name,
                        $entry->date,
                        $entry->task_title ?: 'N/A',
                        $entry->hours_logged,
                        strip_tags( $entry->description ),
                        ucfirst( str_replace( '_', ' ', $entry->status ) )
                    ] );
                }
                break;

            case 'leave':
                fputcsv( $output, [ 'Employee', 'Leave Type', 'Start Date', 'End Date', 'Total Days', 'Reason', 'Status', 'Applied On' ] );
                foreach ( $data['records'] as $record ) {
                    fputcsv( $output, [
                        $record->employee_name,
                        ucfirst( $record->leave_type ),
                        $record->start_date,
                        $record->end_date,
                        $record->total_days,
                        strip_tags( $record->reason ),
                        ucfirst( $record->status ),
                        date( 'Y-m-d H:i', strtotime( $record->created_at ) )
                    ] );
                }
                break;
        }

        fclose( $output );
        exit;
    }

    /**
     * Export report to PDF (Simple HTML-based)
     */
    private static function export_to_pdf( $type, $data ) {
        // For now, we'll generate an HTML version that can be printed to PDF
        // In a production environment, you might want to use a library like TCPDF or mPDF
        
        header( 'Content-Type: text/html; charset=utf-8' );
        
        echo '<!DOCTYPE html>';
        echo '<html><head>';
        echo '<title>' . esc_html( $data['title'] ) . '</title>';
        echo '<style>
            body { font-family: Arial, sans-serif; margin: 40px; }
            h1 { color: #2271b1; }
            table { width: 100%; border-collapse: collapse; margin-top: 20px; }
            th, td { padding: 10px; text-align: left; border-bottom: 1px solid #ddd; }
            th { background-color: #2271b1; color: white; }
            .stats { display: grid; grid-template-columns: repeat(4, 1fr); gap: 20px; margin: 20px 0; }
            .stat-card { padding: 15px; background: #f0f0f1; border-radius: 4px; }
            .stat-number { font-size: 24px; font-weight: bold; color: #2271b1; }
            @media print { button { display: none; } }
        </style>';
        echo '</head><body>';
        echo '<h1>' . esc_html( $data['title'] ) . '</h1>';
        echo '<p>' . esc_html( $data['subtitle'] ) . '</p>';
        echo '<button onclick="window.print()">Print to PDF</button>';
        
        // Display statistics
        if ( isset( $data['stats'] ) ) {
            echo '<div class="stats">';
            foreach ( $data['stats'] as $key => $value ) {
                echo '<div class="stat-card">';
                echo '<div>' . esc_html( ucfirst( str_replace( '_', ' ', $key ) ) ) . '</div>';
                echo '<div class="stat-number">' . esc_html( $value ) . '</div>';
                echo '</div>';
            }
            echo '</div>';
        }

        echo '</body></html>';
        exit;
    }
}
